<?php
require_once __DIR__ . '/config.php';

function ensure_schema(PDO $pdo): void
{
    $stmt = $pdo->query("SELECT DATABASE() AS db");
    $db = $stmt->fetch(PDO::FETCH_ASSOC);
    $dbName = $db ? ($db['db'] ?? '') : '';
    if ($dbName === '') return;

    // mobile_users
    $stmt = $pdo->prepare("SELECT COUNT(*) AS c FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'mobile_users'");
    $stmt->execute([$dbName]);
    if ((int)($stmt->fetch(PDO::FETCH_ASSOC)['c'] ?? 0) === 0) {
        $pdo->exec("
            CREATE TABLE mobile_users (
                id INT UNSIGNED NOT NULL AUTO_INCREMENT,
                phone_prefix VARCHAR(8) NOT NULL,
                phone_number VARCHAR(32) NOT NULL,
                email VARCHAR(190) NULL,
                password_hash CHAR(64) NOT NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                UNIQUE KEY uniq_phone (phone_prefix, phone_number),
                KEY idx_email (email)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
    }

    // devices (boxy serwerowe) – rejestrowane przy pierwszym uruchomieniu
    $stmt = $pdo->prepare("SELECT COUNT(*) AS c FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'devices'");
    $stmt->execute([$dbName]);
    if ((int)($stmt->fetch(PDO::FETCH_ASSOC)['c'] ?? 0) === 0) {
        $pdo->exec("
            CREATE TABLE devices (
                id INT UNSIGNED NOT NULL AUTO_INCREMENT,
                identifier VARCHAR(255) NOT NULL,
                name VARCHAR(128) NOT NULL,
                api_key CHAR(64) NOT NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                UNIQUE KEY uniq_identifier (identifier),
                UNIQUE KEY uniq_api_key (api_key)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
    } else {
        // istniejąca tabela – upewnij się, że jest kolumna identifier
        $stmt = $pdo->prepare("SELECT COUNT(*) AS c FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'devices' AND COLUMN_NAME = 'identifier'");
        $stmt->execute([$dbName]);
        if ((int)($stmt->fetch(PDO::FETCH_ASSOC)['c'] ?? 0) === 0) {
            $pdo->exec("ALTER TABLE devices ADD COLUMN identifier VARCHAR(255) NULL AFTER id");
            $pdo->exec("UPDATE devices SET identifier = CONCAT('legacy-', id) WHERE identifier IS NULL");
            $pdo->exec("ALTER TABLE devices MODIFY COLUMN identifier VARCHAR(255) NOT NULL");
            try {
                $pdo->exec("ALTER TABLE devices ADD UNIQUE KEY uniq_identifier (identifier)");
            } catch (Throwable $e) {}
        }
    }

    // mobile_assignments (który telefon ma jaki plik z którego urządzenia)
    $stmt = $pdo->prepare("SELECT COUNT(*) AS c FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'mobile_assignments'");
    $stmt->execute([$dbName]);
    if ((int)($stmt->fetch(PDO::FETCH_ASSOC)['c'] ?? 0) === 0) {
        $pdo->exec("
            CREATE TABLE mobile_assignments (
                id INT UNSIGNED NOT NULL AUTO_INCREMENT,
                mobile_user_id INT UNSIGNED NOT NULL,
                device_id INT UNSIGNED NOT NULL,
                file_id INT UNSIGNED NOT NULL,
                file_name VARCHAR(255) NOT NULL,
                file_ext VARCHAR(16) NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY idx_user (mobile_user_id),
                KEY idx_device (device_id),
                KEY idx_user_device (mobile_user_id, device_id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
        ");
    }
}

function normalize_phone(string $prefix, string $phone): array
{
    $pfx = trim($prefix);
    $pfx = ltrim($pfx, '+');
    $pfx = preg_replace('/\D+/', '', $pfx);

    $num = preg_replace('/\D+/', '', (string)$phone);

    return [$pfx, $num];
}

function register_mobile_user(PDO $pdo, string $prefix, string $phone, ?string $email, string $password): array
{
    [$pfx, $num] = normalize_phone($prefix, $phone);
    if ($pfx === '' || $num === '') {
        return ['error' => 'invalid_phone'];
    }
    if ($password === '' || strlen($password) < 4) {
        return ['error' => 'weak_password'];
    }

    $stmt = $pdo->prepare("SELECT id FROM mobile_users WHERE phone_prefix = ? AND phone_number = ? LIMIT 1");
    $stmt->execute([$pfx, $num]);
    if ($stmt->fetch()) {
        return ['error' => 'phone_taken'];
    }

    $hash = hash('sha256', $password);
    $stmt = $pdo->prepare("INSERT INTO mobile_users (phone_prefix, phone_number, email, password_hash) VALUES (?,?,?,?)");
    $stmt->execute([$pfx, $num, $email ?: null, $hash]);
    $id = (int)$pdo->lastInsertId();

    return [
        'id' => $id,
        'phone_prefix' => $pfx,
        'phone_number' => $num,
        'email' => $email
    ];
}

function login_mobile_user(PDO $pdo, string $prefix, string $phone, string $password): ?array
{
    [$pfx, $num] = normalize_phone($prefix, $phone);
    if ($pfx === '' || $num === '') {
        return null;
    }

    $stmt = $pdo->prepare("SELECT id, phone_prefix, phone_number, email, password_hash FROM mobile_users WHERE phone_prefix = ? AND phone_number = ? LIMIT 1");
    $stmt->execute([$pfx, $num]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) return null;

    $hash = hash('sha256', $password);
    if (!hash_equals($user['password_hash'], $hash)) {
        return null;
    }

    $_SESSION['mobile_user_id'] = (int)$user['id'];

    return [
        'id' => (int)$user['id'],
        'phone_prefix' => $user['phone_prefix'],
        'phone_number' => $user['phone_number'],
        'email' => $user['email']
    ];
}

function generate_reset_password(PDO $pdo, string $email): ?string
{
    $email = trim($email);
    if ($email === '') return null;

    $stmt = $pdo->prepare("SELECT id FROM mobile_users WHERE email = ? LIMIT 1");
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) return null;

    $plain = substr(bin2hex(random_bytes(4)), 0, 8);
    $hash = hash('sha256', $plain);

    $stmt = $pdo->prepare("UPDATE mobile_users SET password_hash = ? WHERE id = ?");
    $stmt->execute([$hash, (int)$user['id']]);

    // Tutaj w docelowej infrastrukturze wyślesz maila z $plain
    return $plain;
}

function list_assigned_files(PDO $pdo, int $mobileUserId): array
{
    $stmt = $pdo->prepare("
        SELECT ma.id,
               ma.device_id,
               ma.file_id,
               ma.file_name,
               ma.file_ext,
               ma.created_at,
               d.name AS device_name
        FROM mobile_assignments ma
        LEFT JOIN devices d ON d.id = ma.device_id
        WHERE ma.mobile_user_id = ?
        ORDER BY ma.created_at DESC, ma.id DESC
    ");
    $stmt->execute([$mobileUserId]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    return array_map(function ($r) {
        return [
            'id' => (int)$r['id'],
            'device_id' => (int)$r['device_id'],
            'device_name' => $r['device_name'] ?? null,
            'file_id' => (int)$r['file_id'],
            'file_name' => $r['file_name'],
            'file_ext' => $r['file_ext'],
            'created_at' => $r['created_at'],
        ];
    }, $rows);
}

function assign_file_to_phone(PDO $pdo, int $deviceId, string $prefix, string $phone, int $fileId, string $fileName, ?string $fileExt): array
{
    [$pfx, $num] = normalize_phone($prefix, $phone);
    if ($pfx === '' || $num === '') {
        return ['error' => 'invalid_phone'];
    }

    // sprawdź, czy użytkownik istnieje
    $stmt = $pdo->prepare("SELECT id FROM mobile_users WHERE phone_prefix = ? AND phone_number = ? LIMIT 1");
    $stmt->execute([$pfx, $num]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        return ['error' => 'user_not_found'];
    }
    $uid = (int)$row['id'];

    // unikaj duplikatów
    $stmt = $pdo->prepare("
        SELECT id FROM mobile_assignments
        WHERE mobile_user_id = ? AND device_id = ? AND file_id = ?
        LIMIT 1
    ");
    $stmt->execute([$uid, $deviceId, $fileId]);
    if ($stmt->fetch()) {
        return ['ok' => true, 'assignment' => 'exists'];
    }

    $stmt = $pdo->prepare("
        INSERT INTO mobile_assignments (mobile_user_id, device_id, file_id, file_name, file_ext)
        VALUES (?,?,?,?,?)
    ");
    $stmt->execute([$uid, $deviceId, $fileId, $fileName, $fileExt ?: null]);
    $id = (int)$pdo->lastInsertId();

    return [
        'ok' => true,
        'id' => $id
    ];
}

/**
 * Rejestracja urządzenia serwerowego w proxy (przy pierwszym uruchomieniu).
 * Jeśli identifier już istnieje – zwraca istniejący api_key.
 */
function register_device(PDO $pdo, string $identifier, string $name): array
{
    $identifier = trim($identifier);
    $name = trim($name);
    if ($identifier === '' || $name === '') {
        return ['error' => 'missing'];
    }
    if (strlen($identifier) > 255) {
        $identifier = substr($identifier, 0, 255);
    }
    if (strlen($name) > 128) {
        $name = substr($name, 0, 128);
    }

    $stmt = $pdo->prepare("SELECT id, api_key FROM devices WHERE identifier = ? LIMIT 1");
    $stmt->execute([$identifier]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        return [
            'ok' => true,
            'device_id' => (int)$row['id'],
            'api_key' => $row['api_key']
        ];
    }

    $apiKey = bin2hex(random_bytes(32));
    try {
        $stmt = $pdo->prepare("INSERT INTO devices (identifier, name, api_key) VALUES (?, ?, ?)");
        $stmt->execute([$identifier, $name, $apiKey]);
    } catch (Throwable $e) {
        // np. duplikat identifier (race)
        $stmt = $pdo->prepare("SELECT id, api_key FROM devices WHERE identifier = ? LIMIT 1");
        $stmt->execute([$identifier]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) {
            return [
                'ok' => true,
                'device_id' => (int)$row['id'],
                'api_key' => $row['api_key']
            ];
        }
        return ['error' => 'register_failed'];
    }

    return [
        'ok' => true,
        'device_id' => (int)$pdo->lastInsertId(),
        'api_key' => $apiKey
    ];
}

// Crafted with ❤️ by PixelShark.eu

